
% 
% Simple Robot Control with Obstacle Avoidance 
%
% Launch the virtual machine.
% Open terminal, and start the empty Gazebo world: ./start-gazebo-empty.sh.
%
% Versions: 
% v02 - UC Summer School 2023 

% clear memory 
clearvars -except tbot

% -----------------------------------------------------------------------------
% start TurtleBot connection (tbot object), only if required
if ( ~exist("tbot") ) 
    % init by assign the IPs directly by the TurtleBot constructor
    IP_TURTLEBOT = "192.168.1.110";         % virtual machine IP (or robot IP)
    IP_HOST_COMPUTER = "192.168.1.100";     % local machine IP
    tbot = TurtleBot3(IP_TURTLEBOT, IP_HOST_COMPUTER);

    % check if robot's simulation is running   
    if ( ~tbot.isRealRobot() ) 
        % Delete all suported 3D objects  
        tbot.gazeboDeleteAllModels();
        
        % Pause Gazebo physics simulation 
        tbot.gazeboPause();

        % Place 3D Objects ------
        % place cube at (x=2, y=0, orientation=0, edgeSize=1, color='white')
        tbot.gazeboPlace3DCube(2, 0, 0, 1, 'w');    
        %tbot.gazeboPlace3DCube(1.5, 1.5, 0, 1, 'w');
        %tbot.gazeboPlace3DCube(1.5, -1.5, 0, 1, 'w');

        % Resume Gazebo physics simulation
        tbot.gazeboResume();
    end 
end 
% -----------------------------------------------------------------------------


% Reset Turtlebot's 2D pose, place robot at (x,y,theta) -> (0,0,0) 
tbot.resetPose();
pause(1);           % wait 1s 

% init ratecontrol obj - enables to run a loop at a ~ fixed frequency 
r = rateControl(5);     % run at 5Hz

% init trajectory table (pose history information)
trajectory = zeros(500,3);
% init iteration counter 
it = 1;


tic;                        % tic; ... toc;  - allows to measure elapsed time   
while (toc < 25)            % run from 25 seconds
   
    % Read TurtleBot's pose (place and orientation where is the robot in the 2D map)
    [x, y, theta, timestamp] = tbot.readPose();

    % Hold trajectory data
    trajectory(it,:) = [x, y, theta];    
    it = it + 1;                        % update iterations counter

    % ----------------------------
    % Obstacle Detection  
    % ----------------------------
    [detectionFlag, detectionSettings] = tbot.naiveObstacleDetection(0.4, pi/8);
    
    % ----------------------------
    % Display robot + detections 
    % ----------------------------
    figure(1); clf; hold on;            % init window, clear figure and hold plot 
    drawTurtleBot(x, y, theta);         % draw Robot
    plot(trajectory(1:it-1,1), trajectory(1:it-1,2), 'b'); % show trajectory 
    drawObstacleSensor(x, y, theta, detectionFlag, detectionSettings); % draw obstacle sensor 
    quiver(0,0,0.5,0,'r')               % draw arrow for x-axis 
    quiver(0,0,0,0.5,'g')               % draw arrow for y-axis 
    axis([-1, 3, -2, 2])                % the limits for the current axes [xmin xmax ymin ymax]
    grid on;                            % enable grid 
    xlabel('x')                         % show axis labels x/y 
    ylabel('y')
    % adaptive pause
    waitfor(r);


    % ----------------------------
    % Setup Robot's Actions 
    % ----------------------------

    if( detectionFlag )

        break;      % exit loop      

    else 
        % move forward 
        tbot.move(0.1); 
    end 


    
end % end main loop 



% stop robot 
tbot.stop();  




