
%
% uTurtleBot3 Control Class 
% - Micro TurtleBot3 (uTurtleBot3) robot simulator interface. 
% - Follows TurtleBot3 main interface
% 
%       Industrial Robotics | Fundamentals of Robotics and Bionics
%       Department of Mechanical Engineering
%       University of Coimbra, Portugal 
%
%       Pedro Martins
%       Department of Mechanical Engineering
%       University of Coimbra, Portugal
%       pedro.martins@dem.uc.pt
% 
% v01c version - 02/10/2025   
%
% versions:  
% v01a - initial
% v01b - added computeOdometry(.), getICC(.) and getICCRadius(.) 
% v01c - uTurtleBot3 core based in DiffDrive interface


classdef uTurtleBot3 < handle

    properties (Access = private)
        version = 1.3           % version number  
        pose = [0, 0, 0]';      % 2D pose = [x,y,theta]'
        velocity = [0, 0];      % velocity = [v, w]' 
        dt = 0.2                % control loop time interval [default 5Hz]          
        wheelbaseline = 0.16    % wheels separation (distance between wheels) [meters]
        wheelradius = 0.033     % wheel radius [meters]
        iccRadius = 0           % current ICC radius 
        ticks = 0               % timestamp
    end

    methods

    % Constructor (two options)  
    function obj = uTurtleBot3( varargin )
        % Constructor (two options)  
        % obj = uTurtleBot3();
        % obj = uTurtleBot3(dt);
   
         % check input args 
        switch nargin
        case 0
                % dt = 0.2 w/ no input parameters [default]  
        case 1
            dt = varargin{1};                   % assign time interval  

            % revert to default if sampling time defined to be zero or negative
            if(dt <= 0) dt = 0.2; end 

        otherwise
            error("Usage: obj = uTurtleBot3( dt )");
        end
       
    end     % end constructor 


    % Send linear and angular velocity commands. 
    % ->Odometry in evaluted in this function!
    function setVelocity(obj, v, w)
        % Send linear (v) and angular (w) velocity commands.
        % 
        % obj.setVelocity(v, w);
        % 
        % (v,w) are scalars with units in [m/s] and [rads/s], respectively.
        %
        % notes:
        % -> negative linear velocity values move the robot backwards.
        % -> negative angular velocity values rotate the robot clockwise.     
        % -> both velocity values are limited to a maximum [vmax = 0.22 m/s, wmax = 2.84 rad/s].

        % clip (saturate) velocity to maximum values [see turtlebot manual]
        if( abs(v) > 0.22) v = sign(v) * 0.22; end      % [max v = 0.22 m/s]
        if( abs(w) > 2.84) w = sign(w) * 2.84; end      % [max w = 2.84 rad/s]


        % assign linear velocity
        obj.velocity(1) = v; 

        % assign angular velocity
        obj.velocity(2) = w; 

        % update ICC radius
        obj.iccRadius = v / w;

    end


    % update robot's pose (compute odometry)
    function update(obj)
    
        % get linear + angular velocity
        v = obj.velocity(1); 
        w = obj.velocity(2); 

        % linear motion update 
        ds = v * obj.dt; 

        %vr = v + w/2; 
        %vl = v - w/2; 
        % angular motion update
        %dw = (vr - vl) * obj.dt / obj.wheelbaseline;  

        % right wheel traveled distance
        dsr = (v + w/2) * obj.dt; 

        % left wheel traveled distance
        dsl = (v - w/2) * obj.dt; 

        % angular motion update
        dw = (dsr - dsl) / obj.wheelbaseline;  


        % apply kinematic model 
        obj.pose = obj.pose + [ ds * cos( obj.pose(3) + dw/2 );  ds * sin( obj.pose(3) + dw/2 );  dw];

        % normalize angle
        if( abs( obj.pose(3) ) > pi ) obj.pose(3) = atan2( sin( obj.pose(3) ), cos( obj.pose(3) ) ); end 

        % update internal clock
        obj.ticks = obj.ticks + 1;

    end 


 




    % Reads the 2D pose 
    function [x, y, theta, ptimestamp] = readPose(obj)
        % Reads the 2D pose of the robot.
        %
        % [x, y, theta, ptimestamp] = obj.readPose();
        %  
        % returns: 
        % - (x, y) positon [in meters].
        % - (theta) orientation [rad].
        % - ptimestamp reading timestamp (in seconds).


        % compute odometry 
        obj.update();  


        x = obj.pose(1);
        y = obj.pose(2);
        theta = obj.pose(3);
        ptimestamp = obj.ticks * obj.dt;  

    end


    % Set 2D pose to (x,y,theta)
    function setPose(obj, x, y, theta)
        % Sets (overwrites) the 2D pose to (x, y, theta). 
        %
        % obj.setPose(x, y, theta);
        % 
        % - (x, y) positon [in meters].
        % - (theta) orientation [rad].

        obj.pose = [x, y, theta]';
        
    end 


    % Reset pose to origin: (x,y,theta) = (0,0,0) 
    function resetPose(obj)
        % Resets the 2D pose to the origin (x = 0, y = 0, theta = 0).
        %
        % obj.resetPose();
 
        % set pose to origin (forward oriented to x axis)
        obj.setPose(0.0, 0.0, 0.0);       
  
    end 


    % get ICC radius
    function r = getICCRadius(obj)
        r = obj.iccRadius; 
    end

    % return the ICC point (iccx, iccy)
    function [iccx, iccy] = getICC(obj)

        % ICC coordinates 
        iccx = obj.pose(1) - obj.iccRadius * sin( obj.pose(3) ); 
        iccy = obj.pose(2) + obj.iccRadius * cos( obj.pose(3) );

    end


    % Stop robot - send (zero) velocity commands
    function stop(obj)
        % Stop robot. Send (v = 0, w = 0) velocity commands.
        %
        % obj.stop();
        %

        % set velocities v = 0, w = 0 
        obj.setVelocity(0, 0); 

    end


    % Read lidar data
    function [scanMsg, lddata, ldtimestamp] = readLidar(obj, map)
        % Read lidar data
        %
        % [scanMsg, lddata, ldtimestamp] = obj.readLidar( map );
        %
        % receives:
        % map - local map (2D matrix w/ obstacles)  
        % 
        % returns: 
        % - (scanMsg) laser obj struct (ROS message)
        % - (lddata) lidar data struct
        %           - lddata.Ranges - (radial/polar) distance measurement [meters] (360 x 1) vector
        %           - lddata.Angles - angular measurement [radians] (360 x 1) vector
        %           - lddata.Cartesian - X/Y cartesian data (360 x 2) matrix
        % - (ldtimestamp) laser scan reading timestamp (in seconds)
        % 
        % notes:
        % -> All measurements are given in w.r.t. the LIDAR sensor (placed at top of the robot). 
        % -> The lsdata (struct field) arrays could have 'Inf' (or zero) values to represent no laser reflections (representing too near or too far readings). 
        % -> Use getInRangeLidarDataIdx(.) and getOutRangeLidarDataIdx(.) functions to select the desired data. 
        % -> rosPlot(scanMsg) can be used to display. 



        % ---------
        % create empty message
        scanMsg = rosmessage('sensor_msgs/LaserScan'); 

        % init return data struct 
        lddata.Ranges = zeros(360, 1);
        lddata.Angles = zeros(360, 1);
        lddata.Cartesian = zeros(360, 2);


        fprintf("WARNING - readLidar() function not yet implemented...\n"); 



        % timestamp
        ldtimestamp = obj.ticks * obj.dt;  
    
    end



    % Destructor
    % not required

    end     % end methods

end


