
%
% uTurtleBot3 Class: Mobile Robot simulator without the Virtual Machine core.
% - Micro TurtleBot3 (uTurtleBot3) robot simulator interface. 
% - Follows the main TurtleBot3 class layout settings. 
% 
%       Industrial Robotics | Fundamentals of Robotics and Bionics
%       Department of Mechanical Engineering
%       University of Coimbra, Portugal 
%
%       Pedro Martins
%       Department of Mechanical Engineering
%       University of Coimbra, Portugal
%       pedro.martins@dem.uc.pt
% 
% v02a version - 30/10/2025 - see Release Notes.txt for details.   
%

classdef uTurtleBot3 < handle

    properties (Access = private)
        version = 2.1           % version number  
        pose = [0, 0, 0]';      % 2D pose = [x,y,theta]'
        velocity = [0, 0]';     % velocity = [v, w]' 
        dt = 0.2                % control loop sample interval [default f = 5Hz]          
        wheelbaseline = 0.16    % wheels separation (distance between wheels) [meters]
        wheelradius = 0.033     % wheel radius [meters]
        iccRadius = 0           % current ICC radius 
        ticks = 0               % timestamp
        map = zeros(400, 400);  % virtual map 
        mscale = 100;           % map scale factor in pixels (1m -> 100px)
    end

    methods

    % Constructor (two options)  
    function obj = uTurtleBot3( varargin )
        % Constructor (two options)  
        % obj = uTurtleBot3();
        % obj = uTurtleBot3(dt);
   
         % check input args 
        switch nargin
        case 0
                % dt = 0.2 w/ no input parameters [default]  
        case 1
            dt = varargin{1};                   % assign sampling time 

            % revert to default if sampling time defined to be zero or negative
            if(dt <= 0) dt = 0.2; end 

        otherwise
            error("Usage: obj = uTurtleBot3( dt )");
        end
       
    end     % end constructor 


    % get version number
    function vs = getVersion(obj)
        % Get the version number.
        %
        % vs = obj.getVersion();
        % 
        % - returns a scalar value.  

        vs = obj.version; 
    end


    % Send linear and angular velocity commands. 
    % ->Odometry in evaluted in this function!
    function setVelocity(obj, v, w)
        % Send linear (v) and angular (w) velocity commands.
        % 
        % obj.setVelocity(v, w);
        % 
        % (v,w) are scalars with units in [m/s] and [rads/s], respectively.
        %
        % notes:
        % -> negative linear velocity values move the robot backwards.
        % -> negative angular velocity values rotate the robot clockwise.     
        % -> both velocity values are limited to a maximum [vmax = 0.22 m/s, wmax = 2.84 rad/s].

        % clip (saturate) velocity to maximum values [see turtlebot manual]
        if( abs(v) > 0.22) v = sign(v) * 0.22; end      % [max v = 0.22 m/s]
        if( abs(w) > 2.84) w = sign(w) * 2.84; end      % [max w = 2.84 rad/s]


        % assign linear velocity
        obj.velocity(1) = v; 

        % assign angular velocity
        obj.velocity(2) = w; 

        % update ICC radius
        obj.iccRadius = v / w;

    end


    % update robot's pose (compute odometry)
    function update(obj)
    
        % get linear + angular velocity
        v = obj.velocity(1); 
        w = obj.velocity(2); 

        % linear motion update 
        ds = v * obj.dt; 

        %vr = v + w/2; 
        %vl = v - w/2; 
        % angular motion update
        %dw = (vr - vl) * obj.dt / obj.wheelbaseline;  

        % right wheel traveled distance
        dsr = (v + w/2) * obj.dt; 

        % left wheel traveled distance
        dsl = (v - w/2) * obj.dt; 

        % angular motion update
        dw = (dsr - dsl) / obj.wheelbaseline;  


        % apply kinematic model 
        obj.pose = obj.pose + [ ds * cos( obj.pose(3) + dw/2 );  ds * sin( obj.pose(3) + dw/2 );  dw];

        % normalize angle
        if( abs( obj.pose(3) ) > pi ) obj.pose(3) = atan2( sin( obj.pose(3) ), cos( obj.pose(3) ) ); end 

        % update internal clock
        obj.ticks = obj.ticks + 1;

    end 


 




    % Reads the 2D pose 
    function [x, y, theta, ptimestamp] = readPose(obj)
        % Reads the 2D pose of the robot.
        %
        % [x, y, theta, ptimestamp] = obj.readPose();
        %  
        % returns: 
        % - (x, y) positon [in meters].
        % - (theta) orientation [rad].
        % - ptimestamp reading timestamp (in seconds).


        % compute odometry 
        obj.update();  


        x = obj.pose(1);
        y = obj.pose(2);
        theta = obj.pose(3);
        ptimestamp = obj.ticks * obj.dt;  

    end


    % Set 2D pose to (x,y,theta)
    function setPose(obj, x, y, theta)
        % Sets (overwrites) the 2D pose to (x, y, theta). 
        %
        % obj.setPose(x, y, theta);
        % 
        % - (x, y) positon [in meters].
        % - (theta) orientation [rad].

        obj.pose = [x, y, theta]';
        
    end 


    % Reset pose to origin: (x,y,theta) = (0,0,0) 
    function resetPose(obj)
        % Resets the 2D pose to the origin (x = 0, y = 0, theta = 0).
        %
        % obj.resetPose();
 
        % set pose to origin (forward oriented to x axis)
        obj.setPose(0.0, 0.0, 0.0);       
  
    end 


    % Stop robot - send (zero) velocity commands
    function stop(obj)
        % Stop robot. Send (v = 0, w = 0) velocity commands.
        %
        % obj.stop();
        %

        % set velocities v = 0, w = 0 
        obj.setVelocity(0, 0); 

    end


    % Get the Instantaneous Center of Curvature (ICC) radius
    function r = getICCRadius(obj)
        % Get the Instantaneous Center of Curvature (ICC) radius
        %
        % r = obj.getICCRadius();
        %

        r = obj.iccRadius; 
    end


    % Get the Instantaneous Center of Curvature (ICC) point coordinates
    function [iccx, iccy] = getICC(obj)
        % Get the Instantaneous Center of Curvature (ICC) coordinates (iccx, iccy)
        %
        % [iccx, iccy] = obj.getICC();
        %

        % ICC coordinates 
        iccx = obj.pose(1) - obj.iccRadius * sin( obj.pose(3) ); 
        iccy = obj.pose(2) + obj.iccRadius * cos( obj.pose(3) );
    end


    % Load Virtual Map (2D occupation grid) 
    function loadVirtualMap(obj, map, mscale)
        % Load Virtual Map (2D occupation grid) [obstacles = 1] 
        %
        % obj.loadVirtualMap(map, mscale)
        % 
        % inputs: 
        %  - map: 2D matrix (occupation grid) 
        %  - mscale: map scale factor (1 meter = 'mscale' pixels)
        % 

        obj.map = map;              % assign 2D Map
        obj.mscale = mscale;        % map scale factor
    end 



    % Load Virtual Map (2D occupation grid) from image file
    function loadVirtualMapImage(obj, filename)
        % Load Map (2D occupation grid) from image file
        % 
        % obj.loadVirtualMapImage(filename)
        % 
        % input: 
        %  - filename: image file path location 
        %
        % example usage: 
        % >> tbot.loadVirtualMapImage('maps/obstacles_grid1.png');
        % 

        % read image
        image = imread( filename );

        % get image size
        [hm, wm, ch] = size(image);

        % decode map size 
        if( hm == 400 && wm == 400)
            % set map scale ratio (number of pixels that represent 1 meter) 
            mscale = 100;   % grid 1x1 
        else
            error("loadVirtualMapImage(.): Invalid image map (only _grid1 maps are supported)");
        end

        % Convert map into occupation grid format (obstacles = 1)
        map = 1 - double( image(:,:,1) ) ./ 255; 
 
        % load ocupation map
        obj.loadVirtualMap(map, mscale); 

    end 


    % Get the internal 2D Virtual Map of rasterized objects 
    function [vmap, mscale] = getVirtualMap(obj)
        %  Get the internal 2D Virtual Map of rasterized objects 
        %
        % [vmap, mscale] = obj.getVirtualMap();
        % 
        % returns:
        %  - vmap - 2D map (logical matrix) w/ obstables being 1s  
        %  - mscale - map scale: # of pixels that represent 1 meter

        vmap = obj.map;         % internal map
        mscale = obj.mscale;    % map scale (default value: 100) 

    end



    % Read LIDAR data
    function [lddata, ldtimestamp, scanMsg] = readLIDAR(obj, noiseSTD)
        % Read LIDAR data
        %
        % [lddata, ldtimestamp, scanMsg] = obj.getLIDAR( noiseSTD = 0.005 );
        %
        % inputs:
        %  - noiseSTD: additive Gaussian noise (default noiseSTD = 0.0075) [0.75 cm]
        % returns: 
        % - (lddata) lidarScan class obj
        %           - lddata.Ranges - (radial/polar) distance measurement [meters] (360 x 1) vector
        %           - lddata.Angles - angular measurement [radians] (360 x 1) vector
        %           - lddata.Cartesian - X/Y cartesian data (360 x 2) matrix
        % - (ldtimestamp) laser scan reading timestamp (in seconds)
        % - (scanMsg) laser obj struct (ROS message)
        % 
        % notes:
        % -> All measurements are given in w.r.t. the LIDAR sensor (placed at top of the robot). 
        % -> The lsdata (struct field) arrays could have 'Inf' (or zero) values to represent no laser reflections (representing too near or too far readings). 
        % -> rosPlot(scanMsg) can be used to display. 


        % check input parameters 
        if (nargin < 2) noiseSTD = 0.005; end          % noise std = 0.5cm 

        % ---------
     
        % init ranges and angles data
        ranges = zeros(360, 1);
        angles = zeros(360, 1);

        %get LIDAR sensor location (xs, ys)
        xs = obj.pose(1) - 0.0305 * cos( obj.pose(3) );
        ys = obj.pose(2) - 0.0305 * sin( obj.pose(3) );   

        %xs = obj.pose(1);
        %ys = obj.pose(2);
        %theta = obj.pose(3);


        % robot position in the map (in pixels)
        us = xs * obj.mscale;  
        vs = ys * obj.mscale;

        % angular resolution
        alpha = deg2rad( [0:1:360] )';

        % LIDAR max range 
        maxRange = 3.0;
        %minRange = 0.12;

        % get (xt, yt) locations around 
        xt = xs + maxRange * cos( alpha + obj.pose(3) ); 
        yt = ys + maxRange * sin( alpha + obj.pose(3) ); 

        % get map size
        [h, w, c] = size(obj.map);  

        % look over radial scanlines
        for k=1:1:360

            % convert to pixel positions
            u = round( xt(k) * obj.mscale);
            v = round( yt(k) * obj.mscale);

            % apply Bresenham line algorithm 
            [ul, vl] = ubresenham(us, vs, u, v);

            % found flag
            found = 0; 
            
            % basic scan in line
            for idx =1:1:length( ul )

                % check ocupation grid limits 
                if( ul(idx) > 1  && ul(idx) <= h && vl(idx) > 1  && vl(idx) <= w )

                    % test for obstacle in scanline
                    if ( obj.map( ul(idx), vl(idx) ) > 0.75 ) 
                        found = 1;
                        break;  
                    end

                end 
            end 


            if ( found )
                % compute L2 distance to 1st obstacle in scanline (note: idx - 1) + Gaussian noise
                distance = sqrt( (ul(idx - 1)/obj.mscale - xs).^2 + (vl( idx - 1)/obj.mscale - ys).^2 ) + noiseSTD.*randn(1,1);
            else 
                distance = nan;
            end 

            % fill data
            ranges(k) = distance;
            angles(k) = alpha(k);
        end 
    
        % return the 2D LiDAR scan obj
        lddata = lidarScan(ranges, angles); 

        % timestamp
        ldtimestamp = obj.ticks * obj.dt;  
    

        % create empty ROS message
        scanMsg = rosmessage("sensor_msgs/LaserScan", "DataFormat", "struct"); 
        % ROS Message data
        %  MessageType: 'sensor_msgs/LaserScan'
        %  Header: [1×1 struct]
        % AngleMin: 0
        % AngleMax: 6.2832
        % AngleIncrement: 0.0175
        % TimeIncrement: 0
        % ScanTime: 0
        % RangeMin: 0.1200
        % RangeMax: 3.5000
        %  Ranges: [360×1 single]
        % Intensities: [360×1 single]

        % RP LIDAR settings 
        scanMsg.AngleMin = 0;
        scanMsg.AngleMax = 2*pi;
        scanMsg.AngleIncrement = deg2rad(1);    % 1 deg
        scanMsg.RangeMin = 0.12;                % 12 cm
        scanMsg.RangeMax = 3.5;                 % 3.5 m
        scanMsg.Ranges = single( ranges );
        %scanMsg.Intensities = single( ranges );


    end     % end readLIDAR(.)


    % Get in range LIDAR data indexs (valid data - obstacles).
    function [vidx] = getInRangeLIDARIdx(obj, lddata, maxRange)
        % Get in range lidar data indexs (i.e. valid lidar data - obstacles).
        %
        % [vidx] = obj.getInRangeLIDARIdx(lddata, maxRange = 3.5);
        %
        % maxRange - is the maximum admitted LIDAR range value (absolute max value = 3.5) [meters].  
        % Returns the linear indexs of several related arrays: lsdata.Ranges(vidx), or lsdata.Angles(vidx) or lddata.Cartesian(vidx,:).
        %
        % Tip (usage): see demoLIDAR.m


        % check inputs
        if(nargin < 3) maxRange = 3.5; end 

        %Define lidar min and max laser ranges (see: scanMsg.RangeMin and scanMsg.RangeMax)
        minRange = 0.12; 
        % maxRange = 3.5; 

        % return lsdata.'DATATYPE' indexs of laser redings within range (valid obstacles)  
        vidx = find( ~isinf( lddata.Ranges ) & lddata.Ranges >= minRange & lddata.Ranges <= maxRange);  
    end 


    % Get the out of range lidar data indexs (too near or too far readings).
    function [nidx] = getOutRangeLIDARIdx(obj, lddata, maxRange)
        % Get the out of range lidar data indexs (too near or too far readings).
        %
        % [nidx] = obj.getOutRangeLIDARIdx(lddata, maxRange = 3.5);
        %
        % maxRange - is the maximum admitted LIDAR range value (absolute max value = 3.5) [meters].  
        % Returns the linear indexs of lsdata.Angles(nidx).
        %
        % Tip (usage): see demoLIDAR.m


        % check inputs
        if(nargin < 3) maxRange = 3.5; end
        
        %Define lidar min and max laser ranges (see: scanMsg.RangeMin and scanMsg.RangeMax)
        minRange = 0.12; 
        % maxRange = 3.5; 

        % return lsdata.'DATATYPE' indexs of out of range readings (too near or too far readings)
        nidx = find( isinf( lddata.Ranges ) | lddata.Ranges < minRange  | lddata.Ranges > maxRange );
    end 



    % Place 3D Cube in the 2D Virtual Map (raster a square) 
    function uPlace3DCube(obj, x, y, orientation, edgeSize)
        % Place 3D Cube in the 2D Virtual Map (raster a square) 
        % 
        % obj.uPlace3DCube(x, y, orientation = 0, edgeSize = 1);
        %  
        % - (x, y) placement position in the map world [in meters].
        % - (orientation) placement orientation [rad]. Default orientation = 0. 
        % - (edgeSize) cube edge length [in meters]. Default edgeSize = 1. 

        % check input parameters 
        if (nargin < 5) edgeSize = 1; end 
        if (nargin < 4) orientation = 0; end 
        if (nargin < 3) error('uPlace3DCube(.) check inputs'); end  
        if (edgeSize < 0) edgeSize = 1; end 
            

        % get map size
        [h, w] = size(obj.map);

        % map scale (1m -> 100px) 
        mscale = obj.mscale; 

        % update edgeSize
        edgeSize = edgeSize / 2;                % fix  

        % square vertexs [x,y] anti-clockwise
        VTX = [-edgeSize, -edgeSize;  edgeSize, -edgeSize;  edgeSize,  edgeSize;  -edgeSize  edgeSize]' .* mscale; 

        % define 2D rotation matrix
        R = [cos(orientation) -sin(orientation); sin(orientation) cos(orientation)];

        % Apply 2D rotation and translation
        UV = (R * VTX)'; 
        U = UV(:,1) + (x * mscale) - 1; 
        V = UV(:,2) + (y * mscale) - 1;  

        % Raster Mask (w/ swaped U/V)
        mask = poly2mask(V, U, h, w);

        % merge mask with internal virtual map
        obj.map = obj.map + mask;

    end


    % Place 3D Cylinder the 2D Virtual Map (raster a circle) 
    function uPlace3DCylinder(obj, x, y, radius)
        % Place 3D Cylinder the 2D Virtual Map (raster a circle) 
        % 
        % obj.uPlace3DCylinder(x, y, radius = 0.5);
        % 
        % - (x, y) placement position in the map world [in meters].
        % - (radius) Cylinder radius [meters]. Default radius = 0.5. 

        % check input parameters 
        if (nargin < 4) radius = 0.5; end 
        if (nargin < 3) error('uPlace3DCylinder(.) check inputs'); end  
        if (radius <= 0) radius = 0.5; end 
    
        
        % get map size
        [h, w] = size(obj.map);

        % map scale (1m -> 100px) 
        mscale = obj.mscale; 

        % exhaustive version ...
        for i=1:1:h 
            for j=1:1:w
                % circle equation
                if( (x - i/mscale)^2 + (y - j/mscale)^2 <= radius^2  ) 
                    % update internal virtual map
                    obj.map(i, j) = 1; 
                end 
            end
        end

    end


   %Place 3D Cardboard Box the 2D Virtual Map (raster a rectangular region) 
    function uPlace3DCardboardBox(obj, x, y, orientation)
        % Place 3D Cardboard Box the 2D Virtual Map (raster a rectangular region) 
        % 
        % obj.uPlace3DCardboardBox(x, y, orientation);
        %  
        % - (x, y) placement position in the simulator world [in meters].
        % - (orientation) placement orientation [rad].
        % 
        % note: 
        % -> The cardboard box size: 33 x 22 x 30 (length x width x height) (cm).


        % check input parameters 
        if (nargin < 4) orientation = 0; end 
        if (nargin < 3) error('uPlace3DCardboardBox(.) check inputs'); end  


        % get map size
        [h, w] = size(obj.map);

        % map scale (1m -> 100px) 
        mscale = obj.mscale; 

        % The cardboard box size: 33 x 22 x 30 (length x width x height) (cm).
        %length = 0.33 / 2; 
        %width = 0.22 / 2;  
        length = 0.22 / 2;      % fix
        width = 0.33 / 2;  

        % square vertexs [x,y] anti-clockwise
        VTX = [-length -width;  length -width;  length  width;  -length  width]' .* mscale; 

        % define 2D rotation matrix
        R = [cos(orientation) -sin(orientation); sin(orientation) cos(orientation)];

        % Apply 2D rotation and translation
        UV = (R * VTX)'; 
        U = UV(:,1) + (x * mscale) - 1; 
        V = UV(:,2) + (y * mscale) - 1;  

        % Raster Mask (w/ swaped U/V)
        mask = poly2mask(V, U, h, w);

        % merge mask with internal virtual map
        obj.map = obj.map + mask;

    end


    % Destructor
    % not required

    end     % end methods

end















% Bresenham line algorithm 
function [x, y] = ubresenham( x1, y1, x2, y2)

    %Matlab optmized version of Bresenham line algorithm. No loops.
    %Format:
    %               [x y]=bresenham(x1,y1,x2,y2)
    %
    %Input:
    %               (x1,y1): Start position
    %               (x2,y2): End position
    %
    %Output:
    %               x y: the line coordinates from (x1,y1) to (x2,y2)
    %
    %Usage example:
    %               [x y]=bresenham(1,1, 10,-5);
    %               plot(x,y,'or');
    
    x1=round(x1); x2=round(x2);
    y1=round(y1); y2=round(y2);
    dx=abs(x2-x1);
    dy=abs(y2-y1);
    steep=abs(dy)>abs(dx);
    if steep t=dx;dx=dy;dy=t; end
    
    %The main algorithm goes here.
    if dy==0 
        q=zeros(dx+1,1);
    else
        q=[0;diff(mod([floor(dx/2):-dy:-dy*dx+floor(dx/2)]',dx))>=0];
    end
    
    %and ends here.
    
    if steep
        if y1<=y2 y=[y1:y2]'; else y=[y1:-1:y2]'; end
        if x1<=x2 x=x1+cumsum(q);else x=x1-cumsum(q); end
    else
        if x1<=x2 x=[x1:x2]'; else x=[x1:-1:x2]'; end
        if y1<=y2 y=y1+cumsum(q);else y=y1-cumsum(q); end
    end

end