
% 
% Demo - Read from uTurtleBot3 (Micro TurtleBot3) LIDAR
%
% Open MatLab and run the script >> udemoLIDAR.m
%

% clear memory 
clear;

%---------------------------------------------------------
% init uTurtleBot3 interface (no IP configuration required)
%---------------------------------------------------------
% init uTurtleBot3 interface w/ internal sampling time (dt) 
dt = 0.2;                 %  frequency = 1/dt = 5Hz
tbot = uTurtleBot3( dt ); 

% check version
if( tbot.getVersion() < 2.1 ) error('uTurtleBot3 v2a required'); end 

%---------------------------------------------------------
% Place 3D Objects (2 cubes)
%---------------------------------------------------------
% uPlace3DCube(x, y, orientation, edgeSize);
tbot.uPlace3DCube(2, 1.5, 0, 0.5);        % cube at (2, 1.5)
tbot.uPlace3DCube(0.5, 2, 0, 0.25);        % cube at (0.5, 2)

%---------------------------------------------------------
% Get Virtual 2D Map (rasterized objects)
%---------------------------------------------------------
% Get the 2D virtual map of rasterized objects (and scale factor) 
[vmap, mscale] = tbot.getVirtualMap();
% get map size (rows, cols) 
[hm, wm] = size(vmap);
% find (rowm, colm) coordinates of obstacles
[rowm, colm] = find( vmap == 1);




% set Turtlebot's 2D pose (x,y,theta) -> (0.25,0.5,0)
tbot.setPose(0.25, 0.5, 0);

% start by moving the robot forward
tbot.setVelocity(0.1, 0)


% -------------------------------------
%  read LIDAR data w/ fixed timming
% -------------------------------------

% init ratecontrol obj (enables to run a loop at a fixed frequency) 
r = rateControl(5);     % run at 5Hz

% run for 15 seconds: 75 iterations (75 x 1/5 = 15s) 
for it = 1:1:75

    % read TurtleBot pose (and timestamp) 
    [x, y, theta, timestamp] = tbot.readPose();

    % load LIDAR data from robot 
    [lddata, ldtimestamp, scanMsg] = tbot.readLIDAR();

    % Get in range LIDAR data indexs (valid data - obstacles).
    [vidx] = tbot.getInRangeLIDARIdx(lddata);

    % load valid X/Y cartesian LIDAR readings          
    xydata = lddata.Cartesian(vidx,:);


    % Get out of range lidar data indexs (too near or too far readings).
    % [nidx] = tbot.getOutRangeLIDARIdx(lddata);
    % load obstacle free orientations (angles)  
    % freeAng = lddata.Angles(nidx);


    % display data (using regular plots w/ cartesian data)
    figure(1); clf;
    plot(xydata(:,1), xydata(:,2), 'b.', 'MarkerSize', 8)   % draw laser readings
    axis([-4, 4, -4, 4])                % the limits for the current axes [xmin xmax ymin ymax]
    grid on;                            % enable grid 
    xlabel('x')                         % axis labels 
    ylabel('y')
    title('demo LIDAR')

    % display LIDAR data (using ROS toolbox)
    %figure(3); clf;
    %rosPlot( scanMsg ) 


    % ------------------ 
    % Display Virtual Map  
    % ------------------   
    figure(2); clf; hold on;
    plot( rowm./mscale, colm./mscale,'k.')  % plot obstacles 
    drawTurtleBot(x, y, theta);         % draw Robot
    quiver(0,0,1,0,0.5,'r')             % draw arrow for x-axis 
    quiver(0,0,0,1,0.5,'g')             % draw arrow for y-axis 
    axis([-0.1, hm/mscale, -0.1, wm/mscale]) % set limits for the current axes
    grid on;                            % enable grid 
    xlabel('x')                         % axis labels 
    ylabel('y')
    title('virtual map')


    % adaptive pause
    waitfor(r);
end

% check statistics of past execution periods
stats = statistics(r);

% stop robot 
tbot.stop()




